#!/bin/bash
set -e

################## SETUP BEGIN
THREAD_COUNT=$(sysctl hw.ncpu | awk '{print $2}')
HOST_ARC=$( uname -m )
XCODE_ROOT=$( xcode-select -print-path )
OPENSSL_VER=OpenSSL_1_1_1i
################## SETUP END

MACSYSROOT=$XCODE_ROOT/Platforms/MacOSX.platform/Developer
OPENSSL_VER_NAME=${OPENSSL_VER//.//-}
BUILD_DIR="$( cd "$( dirname "./" )" >/dev/null 2>&1 && pwd )"

if [ "$HOST_ARC" = "arm64" ]; then
    BUILD_ARC=arm64
else
    BUILD_ARC=x86_64
fi

if [ ! -d $BUILD_DIR/frameworks ]; then

    if [ ! -d $OPENSSL_VER_NAME ]; then
        echo downloading $OPENSSL_VER ...
        git clone --depth 1 -b $OPENSSL_VER https://github.com/openssl/openssl $OPENSSL_VER_NAME
    fi

    echo building $OPENSSL_VER "(-j$THREAD_COUNT)" ...
    pushd $OPENSSL_VER_NAME

    mkdir -p $BUILD_DIR/build/lib.arm64 $BUILD_DIR/build/lib.x86_64

    # Build for arm64
    if [ "$HOST_ARC" = "arm64" ] || [ ! -f "$BUILD_DIR/build/lib.arm64/libssl.dylib" ]; then
        ./Configure --prefix="$BUILD_DIR/build/arm64" --openssldir="$BUILD_DIR/build/ssl" shared darwin64-arm64-cc
        make clean
        make -j$THREAD_COUNT
        make install
        make clean
        mv $BUILD_DIR/build/arm64/lib/libssl.dylib $BUILD_DIR/build/lib.arm64/
        mv $BUILD_DIR/build/arm64/lib/libcrypto.dylib $BUILD_DIR/build/lib.arm64/
    fi

    # Build for x86_64
    if [ "$HOST_ARC" = "x86_64" ] || [ ! -f "$BUILD_DIR/build/lib.x86_64/libssl.dylib" ]; then
        ./Configure --prefix="$BUILD_DIR/build/x86_64" --openssldir="$BUILD_DIR/build/ssl" shared darwin64-x86_64-cc
        make clean
        make -j$THREAD_COUNT
        make install
        make clean
        mv $BUILD_DIR/build/x86_64/lib/libssl.dylib $BUILD_DIR/build/lib.x86_64/
        mv $BUILD_DIR/build/x86_64/lib/libcrypto.dylib $BUILD_DIR/build/lib.x86_64/
    fi

    mkdir -p $BUILD_DIR/frameworks
    cp -R $BUILD_DIR/build/arm64/include $BUILD_DIR/frameworks/Headers

    # Create universal binaries
    mkdir -p $BUILD_DIR/build/lib.universal
    lipo -create -output $BUILD_DIR/build/lib.universal/libssl.dylib \
        $BUILD_DIR/build/lib.arm64/libssl.dylib $BUILD_DIR/build/lib.x86_64/libssl.dylib
    lipo -create -output $BUILD_DIR/build/lib.universal/libcrypto.dylib \
        $BUILD_DIR/build/lib.arm64/libcrypto.dylib $BUILD_DIR/build/lib.x86_64/libcrypto.dylib

    xcodebuild -create-xcframework \
        -library $BUILD_DIR/build/lib.universal/libssl.dylib \
        -output $BUILD_DIR/frameworks/ssl.xcframework

    xcodebuild -create-xcframework \
        -library $BUILD_DIR/build/lib.universal/libcrypto.dylib \
        -output $BUILD_DIR/frameworks/crypto.xcframework

    popd
fi
