/*
 * Copyright 2025 The WebRTC project authors. All rights reserved.
 *
 * Use of this source code is governed by a BSD-style license
 * that can be found in the LICENSE file in the root of the source
 * tree. An additional intellectual property rights grant can be found
 * in the file PATENTS.  All contributing project authors may
 * be found in the AUTHORS file in the root of the source tree.
 */

#include "video/corruption_detection/evaluation/av1_qp_parser.h"

#include <cstdint>
#include <optional>

#include "test/gmock.h"
#include "test/gtest.h"

namespace webrtc {
namespace {

#define OBU_TEMPORAL_DELIMITER 0x12, 0x0
#define OBU_SEQUENCE_HEADER \
  0xa, 0xa, 0x0, 0x0, 0x0, 0x2, 0x27, 0xfe, 0xff, 0xfc, 0xc0, 0x20
#define OBU_FRAME_1                                                           \
  0x32, 0x93, 0x2, 0x10, 0x0, 0xa8, 0x80, 0x0, 0x3, 0x0, 0x10, 0x10, 0x30,    \
      0x0, 0xd3, 0xc6, 0xc6, 0x82, 0xaa, 0x5e, 0xbf, 0x82, 0xf2, 0xa4, 0xa4,  \
      0x29, 0xab, 0xda, 0xd7, 0x1, 0x5, 0x0, 0xb3, 0xde, 0xa8, 0x6f, 0x8d,    \
      0xbf, 0x1b, 0xa8, 0x25, 0xc3, 0x84, 0x7c, 0x1a, 0x2b, 0x8b, 0x0, 0xff,  \
      0x19, 0x1f, 0x45, 0x7e, 0xe0, 0xbe, 0xe1, 0x3a, 0x63, 0xc2, 0xc6, 0x6e, \
      0xf4, 0xc8, 0xce, 0x11, 0xe1, 0x9f, 0x48, 0x64, 0x72, 0xeb, 0xbb, 0x4f, \
      0xf3, 0x94, 0xb4, 0xb6, 0x9d, 0x4f, 0x4, 0x18, 0x5e, 0x5e, 0x1b, 0x65,  \
      0x49, 0x74, 0x90, 0x13, 0x50, 0xef, 0x8c, 0xb8, 0xe8, 0xd9, 0x8e, 0x9c, \
      0xc9, 0x4d, 0xda, 0x60, 0x6a, 0xa, 0xf9, 0x75, 0xd0, 0x62, 0x69, 0xd,   \
      0xf5, 0xdc, 0xa9, 0xb9, 0x4c, 0x8, 0x9e, 0x33, 0x15, 0xa3, 0xe1, 0x42,  \
      0x0, 0xe2, 0xb0, 0x46, 0xd0, 0xf7, 0xad, 0x55, 0xbc, 0x75, 0xe9, 0xe3,  \
      0x1f, 0xa3, 0x41, 0x11, 0xba, 0xaa, 0x81, 0xf3, 0xcb, 0x82, 0x87, 0x71, \
      0x0, 0xe6, 0xb9, 0x8c, 0xe1, 0xe9, 0xd3, 0x21, 0xcc, 0xcd, 0xe7, 0x12,  \
      0xb9, 0xe, 0x43, 0x6a, 0xa3, 0x76, 0x5c, 0x35, 0x90, 0x45, 0x36, 0x52,  \
      0xb4, 0x2d, 0xa3, 0x55, 0xde, 0x20, 0xf8, 0x80, 0xe1, 0x26, 0x46, 0x1b, \
      0x3f, 0x59, 0xc7, 0x2e, 0x5b, 0x4a, 0x73, 0xf8, 0xb3, 0xf4, 0x62, 0xf4, \
      0xf5, 0xa4, 0xc2, 0xae, 0x9e, 0xa6, 0x9c, 0x10, 0xbb, 0xe1, 0xd6, 0x88, \
      0x75, 0xb9, 0x85, 0x48, 0xe5, 0x7, 0x12, 0xf3, 0x11, 0x85, 0x8e, 0xa2,  \
      0x95, 0x9d, 0xed, 0x50, 0xfb, 0x6, 0x5a, 0x1, 0x37, 0xc4, 0x8e, 0x9e,   \
      0x73, 0x9b, 0x96, 0x64, 0xbd, 0x42, 0xb, 0x80, 0xde, 0x57, 0x86, 0xcb,  \
      0x7d, 0xab, 0x12, 0xb2, 0xcc, 0xe6, 0xea, 0xb5, 0x89, 0xeb, 0x91, 0xb3, \
      0x93, 0xb2, 0x4f, 0x2f, 0x5b, 0xf3, 0x72, 0x12, 0x51, 0x56, 0x75, 0xb3, \
      0xdd, 0x49, 0xb6, 0x5b, 0x77, 0xbe, 0xc5, 0xd7, 0xd4, 0xaf, 0xd6, 0x6b, \
      0x38
#define OBU_FRAME_2                                                          \
  0x32, 0x33, 0x30, 0x3, 0xc3, 0x0, 0xa7, 0x2e, 0x46, 0xa8, 0x80, 0x0, 0x3,  \
      0x0, 0x10, 0x1, 0x0, 0xa0, 0x0, 0xed, 0xb1, 0x51, 0x15, 0x58, 0xc7,    \
      0x69, 0x3, 0x26, 0x35, 0xeb, 0x5a, 0x2d, 0x7a, 0x53, 0x24, 0x26, 0x20, \
      0xa6, 0x11, 0x7, 0x49, 0x76, 0xa3, 0xc7, 0x62, 0xf8, 0x3, 0x32, 0xb0,  \
      0x98, 0x17, 0x3d, 0x80

constexpr uint8_t kCodedFrameAv1Frame1Qp81[] = {
    OBU_TEMPORAL_DELIMITER, OBU_SEQUENCE_HEADER, OBU_FRAME_1};
constexpr uint8_t kCodedFrameAv1Frame2Qp81[] = {OBU_TEMPORAL_DELIMITER,
                                                OBU_FRAME_2};

#define OBU_SEQUENCE_HEADER_SVC_L3T1                                      \
  0x0a, 0x0d, 0x00, 0x27, 0x01, 0x01, 0x80, 0x80, 0x40, 0x40, 0x89, 0xff, \
      0x80, 0x34, 0x01
#define OBU_FRAME1_SPATIAL_LAYER1                                             \
  0x32, 0x31, 0x12, 0x73, 0xc0, 0x07, 0xc0, 0x07, 0xd2, 0x80, 0x02, 0x08,     \
      0x20, 0x84, 0x00, 0x09, 0x00, 0x05, 0x6b, 0x68, 0x4c, 0xef, 0x9e, 0xe5, \
      0x27, 0xff, 0x0d, 0xb1, 0x41, 0xde, 0xc9, 0xdd, 0x3c, 0x52, 0xea, 0xbf, \
      0xaa, 0xf5, 0x8f, 0x57, 0xaa, 0xdd, 0xe5, 0xf1, 0x19, 0x22, 0x3d, 0x5e, \
      0x5a, 0x16, 0x2d
#define OBU_FRAME1_SPATIAL_LAYER2                                             \
  0x36, 0x08, 0x17, 0x31, 0xe0, 0x40, 0x02, 0x00, 0x00, 0xf7, 0xc0, 0x07,     \
      0xc0, 0x07, 0xc0, 0xf8, 0x81, 0xff, 0xff, 0xfe, 0x2c, 0x54, 0xd8, 0x00, \
      0x76, 0x90
#define OBU_FRAME1_SPATIAL_LAYER3                                         \
  0x36, 0x10, 0x11, 0x30, 0xe0, 0x84, 0x04, 0x00, 0x01, 0xf1, 0x03, 0xff, \
      0xff, 0xfc, 0x58, 0xa9, 0xb0, 0x00, 0x8c, 0x70

constexpr uint8_t kL3T1Av1TemporaUnit[] = {
    OBU_TEMPORAL_DELIMITER, OBU_SEQUENCE_HEADER_SVC_L3T1,
    OBU_FRAME1_SPATIAL_LAYER1, OBU_FRAME1_SPATIAL_LAYER2,
    OBU_FRAME1_SPATIAL_LAYER3};
constexpr uint8_t kQpSpatialLayer1 = 40u;
constexpr uint8_t kQpSpatialLayer2 = 241u;
constexpr uint8_t kQpSpatialLayer3 = 241u;

TEST(Av1QpParserTest, ParseQpAv1KeyFrame) {
  Av1QpParser parser;
  std::optional<uint32_t> qp = parser.Parse(kCodedFrameAv1Frame1Qp81);
  EXPECT_THAT(qp, 81u);
}

TEST(Av1QpParserTest, ParseQpAv1DeltaFrameError) {
  Av1QpParser parser_error;
  // When the first frame is skipped the `sequence_header` is not set, hence QP
  // must be unattainable.
  std::optional<uint32_t> qp = parser_error.Parse(kCodedFrameAv1Frame2Qp81);
  EXPECT_FALSE(qp.has_value());
}

TEST(Av1QpParserTest, ParseQpAv1DeltaFrameProper) {
  Av1QpParser parser;
  ASSERT_TRUE(parser.Parse(kCodedFrameAv1Frame1Qp81).has_value());
  std::optional<uint32_t> qp = parser.Parse(kCodedFrameAv1Frame2Qp81);
  EXPECT_THAT(qp, 81u);
}

TEST(Av1QpParserTest, ParseQpAv1SvcL3T1SpatialLayer1) {
  Av1QpParser parser;
  // `operating_point` = 2 would get the QP for the lowest resolution.
  std::optional<uint32_t> qp = parser.Parse(kL3T1Av1TemporaUnit,
                                            /*operating_point=*/2);
  EXPECT_THAT(qp, kQpSpatialLayer1);
}

TEST(Av1QpParserTest, ParseQpAv1SvcL3T1SpatialLayer2) {
  Av1QpParser parser;
  // `operating_point` = 1 would get the QP for mid resolution.
  std::optional<uint32_t> qp = parser.Parse(kL3T1Av1TemporaUnit,
                                            /*operating_point=*/1);
  EXPECT_THAT(qp, kQpSpatialLayer2);
}

TEST(Av1QpParserTest, ParseQpAv1SvcL3T1SpatialLayer3) {
  Av1QpParser parser;
  // `operating_point` = 0 would get the QP for the highest resolution.
  std::optional<uint32_t> qp = parser.Parse(kL3T1Av1TemporaUnit,
                                            /*operating_point=*/0);
  EXPECT_THAT(qp, kQpSpatialLayer3);
}

}  // namespace
}  // namespace webrtc
